﻿#pragma once

#include  "ProcessUI.hpp"
#include  "EventHandle.hpp"
#include  "Invoke.hpp"
#include  "CriticalSectionLock.hpp"
#include  "NonFlickerText.hpp"
#include  <szTime.hpp>
#include  <buffers.hpp>

#include  "Resource.h"

// WPARAM = 関数 void func(void *) のポインタ
// LPARAM = 関数に渡すポインタ引数
const UINT WM_INVOKE = WM_APP + 1;

const int TIMER_ID = 1;
const int TIMER_INTERVAL = 100;

class CMainDialog :
  public CDialogImpl<CMainDialog>,
  public CUpdateUI<CMainDialog>,
  public CWinDataExchange<CMainDialog>,
  public CMessageFilter,
  public CIdleHandler,
  public ProcessUI
{
private:
  EventHandle         *pQuitEvent;
  EventHandle         *pDoneEvent;
  HANDLE              hThread;
  CProgressBarCtrl    progressBar;
  CButton             cancelButton;
  CButton             backgroundButton;
  szstring            processText;
  CComBSTR            containerText;
  CComBSTR            itemText;
  CComBSTR            elapsedText;
  CComBSTR            timeleftText;
  CComBSTR            elapsedCaption;
  CComBSTR            timeleftCaption;
  CNonFlickerTextImpl elapsedImpl;
  CNonFlickerTextImpl timeleftImpl;
  CNonFlickerTextImpl containerImpl;
  CNonFlickerTextImpl itemImpl;
  CRITICAL_SECTION    critSect;
  szpp::Time          started;
  u64                 maxPos;
  u64                 curPos;
  bool                initialized;
  bool                exiting;
  bool                needUpdate;
  bool                isBackground;


  void UpdateCaption()
  {
    if (!initialized || !::IsWindow(m_hWnd))
      return;

    sbuf<szchar, 256> buf;
    _stprintf_s(buf, buf.size(), SZL("%s (%d%%)"), processText.c_str(), (int)(100 * curPos / (double)maxPos));
    SetWindowText(buf);
  }

  void SetTimeText(CComBSTR & text, const szpp::Time &t)
  {
    if (!initialized || !::IsWindow(m_hWnd))
      return;

    szpp::Time adjusted(t);
    if (adjusted.get() < 0)
      adjusted = 0;

    sbuf<szchar, 64> buf;
    _stprintf_s(buf, buf.size(), SZL("%.02d:%.02d:%.02d"), adjusted.Hour(), adjusted.Minute(), adjusted.Second());
    text = buf;
  }

public:
  enum { IDD = IDD_MAIN };

  virtual BOOL PreTranslateMessage(MSG *pMsg)
  {
    return CWindow::IsDialogMessage(pMsg);
  }

  // 前はアイドルで UI を更新していたが、今は使っていない。
  virtual BOOL OnIdle()
  {
    return TRUE;
  }

  BEGIN_UPDATE_UI_MAP(CMainDialog)
  END_UPDATE_UI_MAP()

  BEGIN_MSG_MAP_EX(CMainDialog)
    MSG_WM_INITDIALOG(OnInitDialog)
    MSG_WM_SHOWWINDOW(OnShowWindow)
    MSG_WM_DESTROY(OnDestroy)
    MSG_WM_TIMER(OnTimer)
    MESSAGE_HANDLER(WM_SYSCOMMAND, OnSysCommand) // MSG_WM_SYSCOMMAND マクロはコンパイルエラーになるので
    MESSAGE_HANDLER(WM_INVOKE, OnInvoke)
    COMMAND_ID_HANDLER(IDCANCEL, OnCancel)
    COMMAND_ID_HANDLER(IDC_BACKGROUND, OnBackground)
  END_MSG_MAP()

  BEGIN_DDX_MAP(CMainDialog)
    DDX_CONTROL_HANDLE(IDC_PROGRESS,        progressBar)
    DDX_CONTROL_HANDLE(IDCANCEL,             cancelButton)
    DDX_CONTROL_HANDLE(IDC_BACKGROUND,       backgroundButton)
    DDX_TEXT          (IDC_ELAPSED,          elapsedText)
    DDX_TEXT          (IDC_TIMELEFT,         timeleftText)
    DDX_TEXT          (IDC_ELAPSED_CAPTION,  elapsedCaption)
    DDX_TEXT          (IDC_TIMELEFT_CAPTION, timeleftCaption)
    DDX_TEXT          (IDC_CONTAINER,        containerText)
    DDX_TEXT          (IDC_ITEM,             itemText)
  END_DDX_MAP()

  CMainDialog()
  : pQuitEvent(0), pDoneEvent(0), hThread(0),
    progressBar(), cancelButton(), backgroundButton(), processText(), containerText(), itemText(),
    elapsedText(), timeleftText(), elapsedCaption(), timeleftCaption(),
    elapsedImpl(), timeleftImpl(), containerImpl(), itemImpl(),
    critSect(), started(), maxPos(1), curPos(0),
    initialized(false), exiting(false), needUpdate(false), isBackground(false)
  {
  }

  BOOL OnInitDialog(CWindow wndFocus, LPARAM lInitParam)
  {
    InitializeCriticalSection(&critSect);

    CenterWindow();

    HICON hIcon = (HICON)::LoadImage(theModule.GetResourceInstance(), MAKEINTRESOURCE(IDR_MAINFRAME), 
      IMAGE_ICON, ::GetSystemMetrics(SM_CXICON), ::GetSystemMetrics(SM_CYICON), LR_DEFAULTCOLOR);
    SetIcon(hIcon, TRUE);
    HICON hIconSmall = (HICON)::LoadImage(theModule.GetResourceInstance(), MAKEINTRESOURCE(IDR_MAINFRAME), 
      IMAGE_ICON, ::GetSystemMetrics(SM_CXSMICON), ::GetSystemMetrics(SM_CYSMICON), LR_DEFAULTCOLOR);
    SetIcon(hIconSmall, FALSE);

    CMessageLoop *pLoop = theModule.GetMessageLoop();
    ATLASSERT(pLoop != NULL);
    pLoop->AddMessageFilter(this);
    pLoop->AddIdleHandler(this);

    UIAddChildWindowContainer(m_hWnd);

    // SC_CLOSE を無効にする
    EnableMenuItem(GetSystemMenu(FALSE), SC_CLOSE, MF_DISABLED | MF_BYCOMMAND);

    DoDataExchange(false);

    initialized = true;

    // OnEraseBkgnd で何もしないカスタムスタティッククラスでサブクラス化する
    elapsedImpl.SubclassWindow(GetDlgItem(IDC_ELAPSED));
    timeleftImpl.SubclassWindow(GetDlgItem(IDC_TIMELEFT));
    containerImpl.SubclassWindow(GetDlgItem(IDC_CONTAINER));
    itemImpl.SubclassWindow(GetDlgItem(IDC_ITEM));

    SetTimer(TIMER_ID, TIMER_INTERVAL);

    return TRUE;
  }

  void OnShowWindow(BOOL bShow, UINT nStatus)
  {
    SetMsgHandled(FALSE);

    cancelButton.SetWindowText(SZT("Cancel"));
    backgroundButton.SetWindowText(SZT("Background"));
    elapsedCaption = SZT("Elapsed time:");
    timeleftCaption = SZT("Time left:");

    DoDataExchange(FALSE, IDCANCEL);
    DoDataExchange(FALSE, IDC_BACKGROUND);
    DoDataExchange(FALSE, IDC_ELAPSED_CAPTION);
    DoDataExchange(FALSE, IDC_TIMELEFT_CAPTION);
  }

  LRESULT OnSysCommand(UINT /*uMsg*/, WPARAM wParam, LPARAM /*lParam*/, BOOL &bHandled)
  {
    // Alt + F4 を押したときの SC_CLOSE は無視
    if (SC_CLOSE != wParam)
      bHandled = false;
    return TRUE;
  }

  LRESULT OnDestroy()
  {
    KillTimer(TIMER_ID);

    CMessageLoop* pLoop = theModule.GetMessageLoop();
    ATLASSERT(pLoop != NULL);
    pLoop->RemoveMessageFilter(this);
    pLoop->RemoveIdleHandler(this);

    DeleteCriticalSection(&critSect);
    return 0;
  }

  void OnTimer(UINT_PTR nIDEvent)
  {
    if (TIMER_ID != nIDEvent)
    {
      SetMsgHandled(false);
      return;
    }

    if (needUpdate && ::IsWindow(m_hWnd))
    {
      UpdateCaption();
      needUpdate = false;
    }

    if (pDoneEvent != 0 && !exiting && pDoneEvent->Wait(0))
  		CloseDialog(IDOK);
  }

  LRESULT OnInvoke(UINT /*uMsg*/, WPARAM wParam, LPARAM lParam, BOOL &/*bHandled*/)
  {
    InvokedFunction func = (InvokedFunction)wParam;
    func((void *)lParam);
    return TRUE;
  }

	LRESULT OnCancel(WORD /*wNotifyCode*/, WORD wID, HWND /*hWndCtl*/, BOOL &/*bHandled*/)
	{
    // ユーザーに確認
    {
      CriticalSectionLock lock(&critSect);
      if (IDYES != MessageBox(SZT("Do you really want to cancel the operation?"), SZT("Confirmation"), MB_YESNO | MB_ICONQUESTION))
        return 0;
    }

    exiting = true;
    pQuitEvent->Set();

		CloseDialog(wID);
		return 0;
	}

	void CloseDialog(int nVal)
	{
		DestroyWindow();
		::PostQuitMessage(nVal);
	}

  LRESULT OnBackground(WORD /*wNotifyCode*/, WORD /*wID*/, HWND /*hWndCtl*/, BOOL &/*bHandled*/)
  {
    // マルチスレッドオプションを有効にして 7-Zip コーデックを呼ぶので、ワーカースレッドだけ優先度をいじってもあまり意味はないかも。
    if (0 != hThread)
    {
      if (!isBackground)
      {
        // 現在フォアグラウンドレベルで動作しているのでバックグラウンドに変更
        SetThreadPriority(hThread, THREAD_PRIORITY_BELOW_NORMAL);
        backgroundButton.SetWindowText(SZT("Foreground"));
      }
      else
      {
        // 現在バックグラウンドレベルで動作しているのでフォアグラウンドに戻す
        SetThreadPriority(hThread, THREAD_PRIORITY_NORMAL);
        backgroundButton.SetWindowText(SZT("Background"));
      }
      needUpdate = true;
      isBackground = !isBackground;
    }
    return 0;
  }

  //  ProgressSubscriber の実装

  void NotifyThread(HANDLE hThread)
  {
    this->hThread = hThread;
  }

  void SetEvents(EventHandle *pQuitEvent, EventHandle *pDoneEvent)
  {
    this->pQuitEvent = pQuitEvent;
    this->pDoneEvent = pDoneEvent;
  }

  /// 進行状況の最大値を progressMax - 1 に設定するメソッド。
  /// Remarks:
  ///   進行状況は 0 から始まり、progressMax - 1 で終わることが多いので、進行状況の最大値は progressMax より一つ小さくすることにした。
  void SetMaximum(u64 progressMax)
  {
    if (!::IsWindow(m_hWnd))
      return;

    curPos = 0;
    progressBar.SetPos(0);

    maxPos = progressMax - 1;
    progressBar.SetRange(0, 65535);

    started = szpp::Time::Now();

    needUpdate = true;
  }

  void AddCurrent(u64 delta)
  {
    SetCurrent(curPos + delta);
  }

  void SetCurrent(u64 progressCur)
  {
    if (!::IsWindow(m_hWnd))
      return;

    // 0 <= progressCur < progressMax を満たさなければならないが、それは double 演算の丸めで保証している。
    curPos = max(0, progressCur);

    double ratio = curPos / (double)maxPos;

    // 時間の更新
    szpp::Time cur = szpp::Time::Now();
    szpp::Time elapsed = cur - started;
    szpp::Time timeleft = szpp::Time((s64)(elapsed.get() / ratio)) - elapsed;
    SetTimeText(elapsedText, elapsed);
    if (elapsed.Second() < 5)
      timeleftText = SZT("Estimating...");
    else
      SetTimeText(timeleftText, timeleft);

    // 位置の更新
    int pos = min((int)(65535 * ratio), 65535);
    progressBar.SetPos(pos);

    DoDataExchange(FALSE, IDC_ELAPSED);
    DoDataExchange(FALSE, IDC_TIMELEFT);
    DoDataExchange(FALSE, IDC_PROGRESS);

    UpdateCaption();
  }
  
  void SetProcess(const szstring &process)
  {
    processText.assign(process);
    UpdateCaption();
  }

  void SetItem(const szstring &container, const szstring &item)
  {
    containerText = container.c_str();
    itemText = item.c_str();

    DoDataExchange(FALSE, IDC_CONTAINER);
    DoDataExchange(FALSE, IDC_ITEM);
  }

  // .NET Framework の Invoke の真似事
  LRESULT Invoke(InvokedFunction func, void * param)
  {
    CriticalSectionLock lock(&critSect);
    return SendMessage(m_hWnd, WM_INVOKE, (WPARAM)func, (LPARAM)param);
  }
};
